package mybatis.test;

import java.sql.Statement;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.TimeZone;
import static java.time.Month.MARCH;

import org.apache.ibatis.io.Resources;
import org.apache.ibatis.session.SqlSession;
import org.apache.ibatis.session.SqlSessionFactoryBuilder;
import org.junit.Before;
import org.junit.Test;
import static org.junit.Assert.assertEquals;



public class LocalDateTimeTest {

  static {
  }

  private SqlSession sqlSession;
  private TestMapper mapper;

  @Before
  public void setup() throws Exception {
    sqlSession =
        new SqlSessionFactoryBuilder().build(Resources.getResourceAsStream("mybatis-config.xml")).openSession(true);

    mapper = sqlSession.getMapper(TestMapper.class);

    try (Statement statement = sqlSession.getConnection().createStatement()) {
      statement.execute("DROP TABLE IF EXISTS ldt_test");
      statement.execute("CREATE TABLE ldt_test(ldt timestamp without time zone)");
    }
  }

  @Test
  public void testLocalDateTime() {

    // Force into Pacific Time timezone
    TimeZone.setDefault(TimeZone.getTimeZone("America/Los_Angeles"));

    // This following time is a valid "local date time" to be stored and retrieved, it should be unaltered when
    // inserted and selected from the database as a "local date time"
    //
    // Note: the time itself does not exist in Pacific Time (our selected time zone), since it during the
    // "time hole" created by setting clocks forward an hour at 2 am (thus becoming immediately 3 am) for
    // Daylight Saving
    //
    // If mybatis forces conversion to java.sql.Timestamp (which requires a timezone) it will use the current default
    // timezone. In this case it causes the value to change on its way into or out of the database.
    //
    // If mybatis uses LocalDateTime feature of JDBC 4.2 the value will not change on its way into or out of the
    // database.

    LocalDateTime value =
        LocalDateTime.of(LocalDate.of(2019, MARCH, 10), LocalTime.of(2, 30));
    ZonedDateTime value2 = value.atZone(ZoneId.of("America/Los_Angeles"));

    TestMapper mapper = sqlSession.getMapper(TestMapper.class);

    mapper.insertLocalDateTime(value);

    LocalDateTime found = mapper.selectLocalDateTime();
    String foundRaw = mapper.selectRaw();

    // Will currently fail (3:30 gets inserted due to "fixing" time in Pacific time zone
    assertEquals(value, found);
    assertEquals("2019-03-10 02:30:00", foundRaw);
  }

}
