import SwiftUI
import ComposableArchitecture

struct RootFeature: Reducer {
  struct Destination: Reducer {
    enum State: Equatable {
      case sheet(SheetFeature.State)
      case fullScreenCover(FullScreenCoverFeature.State)
    }

    enum Action {
      case sheet(SheetFeature.Action)
      case fullScreenCover(FullScreenCoverFeature.Action)
    }

    var body: some ReducerOf<Self> {
      Scope(
        state: /State.sheet,
        action: /Action.sheet,
        child: SheetFeature.init
      )
      Scope(
        state: /State.fullScreenCover,
        action: /Action.fullScreenCover,
        child: FullScreenCoverFeature.init
      )
    }
  }

  struct State: Equatable {
    @PresentationState var destination: Destination.State?
  }

  enum Action {
    case presentSheet
    case destination(PresentationAction<Destination.Action>)
  }

  var body: some ReducerOf<Self> {
    Reduce { state, action in
      switch action {
      case .presentSheet:
        state.destination = .sheet(SheetFeature.State())
        return .none

      case .destination(.presented(.sheet(.dismissButtonTapped))):
        state.destination = .fullScreenCover(FullScreenCoverFeature.State())
        return .none

      case .destination:
        return .none
      }
    }
    .ifLet(\.$destination, action: /Action.destination, destination: Destination.init)
  }
}

struct RootView: View {
  let store: StoreOf<RootFeature>

  var body: some View {
    VStack(alignment: .center) {
      Text("Root")
        .font(.title)
      Button("Show sheet") {
        store.send(.presentSheet)
      }
      .buttonStyle(.borderedProminent)
    }
    .sheet(
      store: store.scope(state: \.$destination, action: RootFeature.Action.destination),
      state: /RootFeature.Destination.State.sheet,
      action: RootFeature.Destination.Action.sheet,
      content: SheetView.init
    )
    .fullScreenCover(
      store: store.scope(state: \.$destination, action: RootFeature.Action.destination),
      state: /RootFeature.Destination.State.fullScreenCover,
      action: RootFeature.Destination.Action.fullScreenCover,
      content: FullScreenCoverView.init
    )
  }
}
